<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use App\Notifications\Channels\FirebaseChannel;


class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     */
    public function register(): void
    {

    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        if (env('APP_ENV') === 'production') {
            \URL::forceScheme('https'); // Correct usage of URL facade
        }

        // Register custom notification channels
        Notification::extend('fcm', function ($app) {
            return new FirebaseChannel();
        });

        try {
            // Attempt to retrieve the timezone from the settings table
            $timezoneSetting = DB::table('app_settings')->where('key', 'timezone')->first();

            if ($timezoneSetting && $timezoneSetting->value) {
                // If the timezone setting exists, set the timezone dynamically
                config(['app.timezone' => $timezoneSetting->value]);
                date_default_timezone_set($timezoneSetting->value);
            } else {
                // If the timezone setting doesn't exist or is empty, fallback to a default timezone
                config(['app.timezone' => 'Asia/Kolkata']);
                date_default_timezone_set('Asia/Kolkata');
            }

            // Wrap database queries in a try-catch block to handle potential exceptions
            $mailSettings = DB::table('app_settings')
                ->whereIn('key', [
                    'mail_driver',
                    'mail_host',
                    'mail_port',
                    'mail_username',
                    'mail_password',
                    'mail_encryption',
                    'mail_from_address',
                    'mail_from_name'
                ])
                ->pluck('value', 'key')
                ->toArray();

            // Set mail configuration values
            Config::set('mail.driver', $mailSettings['mail_driver'] ?? 'smtp');
            Config::set('mail.host', $mailSettings['mail_host'] ?? 'smtp.mailtrap.io');
            Config::set('mail.port', $mailSettings['mail_port'] ?? 587);
            Config::set('mail.encryption', $mailSettings['mail_encryption'] ?? 'tls');
            Config::set('mail.username', $mailSettings['mail_username'] ?? null);
            Config::set('mail.password', $mailSettings['mail_password'] ?? null);
            Config::set('mail.from.address', $mailSettings['mail_from_address'] ?? 'example@example.com');
            Config::set('mail.from.name', $mailSettings['mail_from_name'] ?? 'Example');
        } catch (\Exception $e) {
            // Log the exception or handle it according to your application's needs
            \Log::error('Error in AppServiceProvider: ' . $e->getMessage());
        }
    }
}
