<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use DB;

class DeliveryAreas extends Model
{
    use HasFactory;

    public $fillable = [
        'zip_code',
        'country',
        'location_name',
        'city',
        'region',
        'status',
        'latitude',
        'longitude',
        'city_id',
        'state_id',
    ];
    
    /**
     * Get the state that owns the delivery area.
     */
    public function state()
    {
        return $this->belongsTo(State::class, 'state_id', 'id');
    }

    public function cities()
    {
        return $this->belongsTo(City::class, 'city_id', 'id');
    }
    
    public function city()
    {
        return $this->belongsTo(City::class, 'city_id', 'id');
    }

    public static function getLists()
    {
        try {
          $query = new Self;

            if(isset($_GET['search']) && $_GET['search']!=''){
                $keyword = $_GET['search'];
                $query = $query->where(function($q) use ($keyword){
                    $q->where(DB::raw('LOWER(location_name)'), 'like', '%'.strtolower($keyword). '%')
                    ->orWhere(DB::raw('zip_code'),'like','%'.strtolower($keyword). '%');
                });
            }
            
            if(isset($_GET['state_id']) && $_GET['state_id']!=''){
                $state_id = $_GET['state_id'];
                $query = $query->where('state_id',$state_id);
            }
            
            if(isset($_GET['city_id']) && $_GET['city_id']!=''){
                $city_id = $_GET['city_id'];
                $query = $query->where('city_id',$city_id);
            }

            if(isset($_GET['status']) && $_GET['status']!=''){
                if($_GET['status'] == 'active'){
                $status = 1;
                }else{
                $status = 0;
                }
                $query = $query->where('status',$status);
            }

          $data = $query->orderBy('id','DESC')->with(['state', 'city'])->paginate(config('constant.paginate.num_per_page'));
          return $data;
        }catch (\Exception $e) {
            return ['status' => false, 'message' => $e->getMessage() . ' '. $e->getLine() . ' '. $e->getFile()];
        }
    }

    public static function getAllowedCities()
    {
        return self::pluck('city')->unique()->toArray();
    }

    /**
     * Get all allowed zip codes from the DeliveryAreas table.
     *
     * @return array
     */
    public static function getAllowedZipCodes()
    {
        return self::pluck('zip_code')->unique()->toArray();
    }

    /**
     * Get all allowed countries from the DeliveryAreas table.
     *
     * @return array
     */
    public static function getAllowedCountries()
    {
        return self::pluck('country')->unique()->toArray();
    }

    /**
     * Validates the delivery address based on city, zip code, country, and coordinates.
     *
     * @param string $zipCode
     * @param string $city
     * @param string $country
     * @param float|null $latitude
     * @param float|null $longitude
     * @return bool
     */
    public static function isValidDeliveryAddress($zipCode, $city, $country, $latitude = null, $longitude = null)
    {
        $allowedCities = self::getAllowedCities();
        $allowedZipCodes = self::getAllowedZipCodes();
        $allowedCountries = self::getAllowedCountries();

        if (in_array($city, $allowedCities)) {
            if (empty($zipCode)) {
                return true;
            }else if (in_array($zipCode, $allowedZipCodes) || in_array($city, $allowedCities)) {
                return true;
            }else{
                return false;
            }
        }

        if (in_array($zipCode, $allowedZipCodes)) {
            return true;
        }
        if (!in_array($country, $allowedCountries)) {
            return false;
        }

        if (!self::isLocationAllowed($latitude, $longitude)) {
            return false;
        }
        return false;
    }

}
