<?php

namespace App\Http\Controllers;
use anlutro\LaravelSettings\Facade as Setting;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Cart;
use App\Models\Coupon;
use App\Models\CouponTranslation;
use App\Models\NonWorkingDays;
use App\Models\ProductTranslation;
use App\Models\Languages;
use App\Models\ProductImage;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;


class ProductController extends Controller
{
    private $language;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $products = Product::getLists();

        $filter = array();
        $filter['products'] = Product::get();

        // Add the new status to the filters
        $filter['status'] = \Helpers::getStatusType();

        return view('products.index',compact('products'))->with('filter',$filter);
    }



    /**
     * Show the form for adding the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function addProduct()
    {
        return view('products.add');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => [
                'required',
                Rule::unique('products')->whereNull('deleted_at')
            ],
            'price' => 'required|numeric|min:1',
        ]);

        $input = $request->all();
        unset($input['_token']);
        if($input['discount_price']!=''){
            if($input['discount_price']>$input['price']){
                return redirect()->back()->with('error',__('lang.message_discount_price_should_be_less_then_or_equal_to_base_price'));
            }
        }


        if(isset($input['image']) && $input['image']!=''){
            $image = $input['image'];
            unset($input['image']);
        }
        
        $input['created_at'] = date('Y-m-d H:i:s');

        $entry_id = Product::insertGetId($input);
        if($entry_id){
            if(isset($image) && $image!=''){
                foreach($image as $image_data){                            
                    $uploadImage = \Helpers::uploadFilesAfterResizeCompressOriginalName($image_data,'images');
                    if($uploadImage['status']==true){
                        $postArr = array(
                            'product_id'=> $entry_id,
                            'image'=> $uploadImage['file_name'],
                            'created_at'=>date('Y-m-d H:i:s')
                        );
                        $image = ProductImage::insertGetId($postArr);
                        Product::where('id', $entry_id)->update(['image' => $uploadImage['file_name']]);
                    } 
                }
            }
        }
        return redirect()->route('products.index')
                        ->with('success',__('lang.message_record_created_successfully'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $product = Product::find($id);
    
        // Fetch related images
        $images = ProductImage::where('product_id', $id)
            ->orderBy('order', 'ASC')
            ->get();
    
        // Process images with fallback logic
        $processedImages = [];
        if ($images->isNotEmpty()) {
            foreach ($images as $image) {
                $processedImages[] = [
                    'id' => $image->id,
                    'image' => $image->image ?: $product->image, // Use fallback to product image
                ];
            }
        } else {
            $processedImages[] = [
                'id' => null,
                'image' => $product->image, // Use product image as fallback
            ];
        }
    
        // Pass data to the view
        $data = [
            'product' => $product,
            'images' => $processedImages, // Processed images with fallback
        ];
    
        return view('products.edit', $data);
    }


    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        $validatedData = $request->validate([
            'name' => [
                'required',
                Rule::unique('products')->ignore($id)->whereNull('deleted_at')
            ],
            'price' => 'required|numeric|min:1',
        ]);

        $input = $request->all();
        $product = Product::find($id);

        if($input['discount_price']!=''){
            if($input['discount_price']>$input['price']){
                return redirect()->back()->with('error',__('lang.message_discount_price_should_be_less_then_or_equal_to_base_price'));
            }
        }

        if(isset($input['image']) && $input['image']!=''){
            $image = $input['image'];
            unset($input['image']);
        }  
        
        
        $input['updated_at'] = date('Y-m-d H:i:s');
        $product->update($input);

        $removedImages = json_decode($input['removed_images'], true);

        if (!empty($removedImages)) {
            foreach ($removedImages as $imageName) {
                // Find and delete the image record from the database
                $Image = ProductImage::where('image', $imageName)->first();
                if ($Image) {
                    $Image->delete();
                }
            }
        }
        
        unset($input['removed_images']);

        if(isset($image) && $image!=''){
            foreach($image as $image_data){                            
                $uploadImage = \Helpers::uploadFilesAfterResizeCompressOriginalName($image_data,'images');
                if($uploadImage['status']==true){
                    $postArr = array(
                        'product_id'=> $id,
                        'image'=> $uploadImage['file_name'],
                        'updated_at'=> date('Y-m-d H:i:s')
                    );
                    $image = ProductImage::insertGetId($postArr);
                    Product::where('id', $id)->update(['image' => $uploadImage['file_name']]);
                } 
            }
        }
        
        $existingImages = ProductImage::where('product_id', $id)->count();

        if ($existingImages == 0 && !empty($product->image)) {
            ProductImage::create([
                'product_id' => $id,
                'image' => $product->image,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        return redirect()->route('products.index')
                        ->with('success',__('lang.message_record_updated_successfully'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        Product::find($id)->delete();
        return redirect()->route('products.index')
                        ->with('success',__('lang.message_record_deleted_successfully'));
    }

    public function translation($id)
    {
        $data = Product::find($id);
        $languages = Languages::where('status',1)->get();
        foreach ($languages as $row) {
            $row->details = ProductTranslation::where('product_id',$id)->where('lang_code',$row->language_code)->first();
            if(!$row->details){
                $postData = array(
                    'product_id' => $id,
                    'lang_code' => $row->language_code,
                    'name' => $data->name,
                    'description' => $data->description,
                    'created_at' => date("Y-m-d H:i:s")
                );
                ProductTranslation::insert($postData);
                $row->details = ProductTranslation::where('product_id',$id)->where('lang_code',$row->language_code)->first();
            }
        }
        return view('products.translation',compact('data'))->with('languages',$languages);
    }

    public function updateTranslation($id,Request $request)
    {
        $post = $request->all();

        request()->validate([
            'language_code' => 'required',
            'name' => 'required',
            'description' => 'required',
        ]);
        $input = $request->all();

        for ($i=0; $i < count($input['language_code']); $i++) {
            if($input['language_code'][$i] == 'en'){
                $updateData = array(
                    'name' => $input['name'][$i],
                    'description'  => $input['description'][$i]
                );
                Product::where('id',$id)->update($updateData);
            }

            $postData = array(
                'lang_code' => $input['language_code'][$i],
                'name' => $input['name'][$i],
                'description' => $input['description'][$i],
                'updated_at' => date("Y-m-d H:i:s")
            );
            ProductTranslation::where('id',$input['translation_id'][$i])->update($postData);
        }
        return redirect()->route('products.index')
                        ->with('success',__('lang.message_translation_updated_successfully'));
    }

    public function changeStatus($id){
        $data = Product::find($id);
        if($data->status == 1){
            $data->update(['status' => 0]);
        }else{
            $data->update(['status' => 1]);
        }
        return redirect()->route('products.index')
                        ->with('success',__('lang.message_status_updated_successfully'));
    }


    public function translateByThirdParty(Request $request)
    {
        $post = $request->all();
        $data = \Helpers::translate($post['text'],$post['name']);
        return response($data);
    }

    public function getProductDetails(Request $request)
    {
        $user = auth()->user();
        if (!$user) {
            return response()->json([
                'status' => 'error',
                'message' => 'Please sign in to add products to the cart.',
                'is_cart_exist' => null
            ]);
        }
        $cart = Cart::where([
            'user_id' => $user->id,
            'product_id' => $request->id
        ])->exists();
        if ($cart) {
            return response()->json([
                'status' => 'error',
                'message' => 'This product is already in the cart. Please go to the cart to edit items.',
                'is_cart_exist' => 1
            ]);
        }
        $product = Product::find($request->id);
        $quantity=($product->minimum_order_quantity>0)?$product->minimum_order_quantity:1;
        $price=$product->discount_price ?? $product->price;
        $total_amount=$quantity*$price;
        //dd($product);
        if (!$product) {
            return response()->json([
                'status' => 'error',
                'message' => 'Product not found.',
                'is_cart_exist' => 0
            ]);
        }

        /* for cart popup */
        $response = array();
        $cartData = Cart::where('user_id',$user->id)->with('product')->get();
        $product_details=array(
            'id'=>-1,
            "user_id" => $user->id,
            "product_id" => $request->id,
            "product_name" => $product->name,
            "product_image"=>$product->image,
            "quantity" => $quantity,
            "per_item_amount" => $price,
            "per_delivery_amount" => $total_amount,
            "total_delivery_amount" => $total_amount,
            "no_of_delivery" =>1,
            "delivery_dates" => date("Y-m-d", strtotime('+1 day')),
            "order_frequency" => 'once',
            "days" => "",
            "start_date" => date("Y-m-d", strtotime('+1 day')),
            "end_date" => date("Y-m-d", strtotime('+1 days')),
        );
        $total = array(
            'subtotal' => 0,
            'tax' => 0,
            'delivery_fee' => 0,
            'is_coupon_applied' => 0,
            'promotional_disount' => 0,
            'coupon_code' => '',
            'final_amount' => 0
        );

        $holdaysDate = NonWorkingDays::where('status',1)->get();
        $html = view('product_details_modal', ['holdaysDate' => $holdaysDate,'product_details'=>$product_details, 'product'=>$product])->render();
        // Return the product details
        return response()->json([
            'status' => 'success',
            'name' => $product->name,
            'image' => asset('app-assets/images/products/' . $product->image),
            'price' => \Helpers::currency($product->discount_price > 0 ? $product->discount_price : $product->price),
            'more_details' => $product,
            'is_cart_exist' => 0,
            'html'=>$html
        ]);

    }
}