<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\UserWallet;
use App\Models\OrderStatus;
use App\Models\OrderProductsDelivery;
use App\Models\User;
use App\Models\Product;
use App\Models\OrderProduct;
use App\Models\OrderCancelRequest;
use App\Models\UserAuthLogin;
use App\Models\Notification;
use App\Models\Driver;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Response;
use App\Exports\OrdersExport;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\Excel as ExcelType;
use App\Exports\RevenueExport;
use App\Exports\OrdersPdfExport;
use App\Exports\RevenuePdfExport;
use PDF;
use Illuminate\Support\Facades\Date;

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $data = Order::getLists();
        $filter = array();
        $filter['user'] = User::where('id','!=',1)->where('type','user')->get();
        $filter['order_status'] = OrderStatus::get();
    
        // Add the new payment method to the filters
        $filter['payment_methods'] = ['cash','wallet','razorpay', 'paypal', 'stripe','instamojo','phonepe','flutterwave','paystack'];
        // , 'paytm', 'phonepe', 'instamojo', 'flutterwave','paystack'
    
        // Add the new payment status to the filters
        $filter['payment_statuses'] = ['Paid', 'Processing','Waiting for Client','Failed'];
    
        return view('orders.index',compact('data'))->with('filter',$filter);
    }



    /**
     * Display the specified Order.
     *
     * @param int $id
     */

    public function show(Request $request, $id)
    {
        $data = Order::where('id',$id)->with('product_orders')->with('order_status')->with('payment')->with('delivery_address')->with('time_slots')->first();
        if (empty($data)) {
            Session::flash('error', 'Order not found!');
            return redirect(route('orders.index'));
        }
        $total_wallet = UserWallet::getWalletbalance($data->user_id);
        $orderStatus = OrderStatus::pluck('status', 'id');

        return view('orders.show', compact('data'))->with("total_wallet",$total_wallet)->with("orderStatus",$orderStatus);
    }

    public function edit(Request $request, $id)
    {
        $data = Order::where('id',$id)->with('product_orders')->with('order_status')->with('payment')->with('delivery_address')->with('time_slots')->first();
        if (empty($data)) {
            Session::flash('error', 'Order not found!');
            return redirect(route('orders.index'));
        }

        $total_wallet = UserWallet::getWalletbalance($data->user_id);
        $orderStatus = OrderStatus::pluck('status', 'id');

        return view('orders.edit', compact('data'))->with("total_wallet",$total_wallet)->with("orderStatus",$orderStatus);
    }

    public function update(Request $request, $id)
    {
        $post = $request->all();

        $data = Order::find($id);
        $input = array(
            'order_status_id' => $post['order_status_id'],
            'updated_at' => date("Y-m-d H:i:s"),
        );
        $data->update($input);

        return redirect()->route('orders.index')
                        ->with('success',__('lang.message_status_updated_successfully'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        // Try to find the order by ID
        $order = Order::find($id);

        // Check if the order exists
        if (!$order) {
            return redirect()->route('orders.index')
                            ->with('error', __('lang.message_order_not_found'));
        }

        // Attempt to delete the order and handle any potential exceptions
        try {
            $order->delete();
            return redirect()->route('orders.index')
                            ->with('success', __('lang.message_record_deleted_successfully'));
        } catch (\Exception $e) {

            return redirect()->route('orders.index')
                            ->with('error', __('lang.message_something_went_wrong'));
        }
    }



    /**
     * Display a listing of the delivery.
     *
     * @return \Illuminate\Http\Response
     */
    public function delivery(Request $request)
    {
        $post = $request->all();
        $data = OrderProductsDelivery::getLists($post);
        $filter = array();
        $filter['user'] = User::where('id','!=',1)->where('type','user')->get();
        $filter['order_status'] = OrderStatus::get();
        $filter['drivers'] = User::where('type','driver')->where('status',1)->get();
        $filter['products'] = Product::where('status','=',1)->get();
        return view('orders.delivery',compact('data'))->with('filter',$filter);
    }

    /**
     * Update the specified Order in storage.
     *
     * @param int $id
     * @param Request $request
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateOrderDeliveryStatus(Request $request)
    {
        $input = $request->all();
        $id = $input['order_item_id'];
        $oldOrder = OrderProductsDelivery::where('id',$id)->first();
        if (empty($oldOrder)) {
            return $this->sendError('Order not found');
        }

        try {
            OrderProductsDelivery::where('id',$id)->update(['order_status_id'=>$input['order_status_id']]);
            $order = OrderProductsDelivery::find($id);
            Order::find($order->order_id)->update(['order_status_id'=>$input['order_status_id']]);
            OrderProductsDelivery::where(["user_id" => $order->user_id,"order_id" => $order->order_id,"product_order_id" => $order->product_order_id,"product_id" => $order->product_id])->update(['order_status_id'=>$input['order_status_id']]);

            $notificationPost = array(
                'user_id' => $oldOrder->user_id,
                'image' => 'en',
                'title' => "Order Status",
                'message' => "Order Status updated successfully.",
                'created_at' => date("Y-m-d H:i:s"),
                'read_status' => 0,
                'order_id' => $oldOrder->order_id,
            );
            Notification::insert($notificationPost);
            $tokens = array();
            $player_ids = array();
            $users = UserAuthLogin::where('user_id',$oldOrder->user_id)->get();

            if(count($users)>0){
                foreach ($users as $row) {
                    if(setting('is_enable_one_signal')){
                        if($row->player_id!=null){
                            array_push($player_ids,$row->player_id);
                        }
                    }else{
                        if($row->fcm_token!=null){
                            array_push($tokens,$row->fcm_token);
                        }
                    }
                }
            }

            if(setting('is_enable_one_signal')){
                \Helpers::sendNotification($player_ids,"Order Status updated  successfully.","Order Status",$key='',false,$id);
            }else{
                \Helpers::sendNotification($tokens,"Order Status updated  successfully.","Order Status",$key='',false,$id);
            }
            
            $orderData = Order::where('id',$order->order_id)->first();
            
            if($input['order_status_id'] == 10){
                if($orderData){
                    $payment_method = $orderData->payment_method;
                    if($payment_method == 'cash'){
                        
                        $opData = OrderProduct::where('id',$oldOrder->product_order_id)->first();
                    
                        $wallertData = [
                            'user_id' => $oldOrder->user_id,
                            'type' => 'credit',
                            'amount' => $opData->per_delivery_amount,
                            ];
                        
                        UserWallet::insert($wallertData);
                    }
                    
                }
            }

            return $this->sendResponse([],__('lang.message_status_updated_successfully'));

        } catch (ValidatorException $e) {
            return $this->sendError($e->getMessage() .' '. $e->getLine() . ' '. $e->getFile());
        }
    }

    public function bulkAssignDriverForDeliver(Request $request)
    {
        $input = $request->all();
        $id = explode(",",$input['order_item_id']);

        try {
            $orderPost = array(
                'driver_id' => $input['driver_id']
            );
            OrderProductsDelivery::whereIn('id',$id)->update($orderPost);

            $key = setting('fcm_key');

            foreach($id as $row){
                $orderProduct = OrderProductsDelivery::where('id',$row)->first();
                $notificationPost = array(
                    'user_id' => $input['driver_id'],
                    'order_id' => $orderProduct->order_id,
                    'order_item_id' => $row,
                    'lang_code' => 'en',
                    'title' => "Order Assigned",
                    'message' => "Order Assigned successfully.",
                    'created_at' => date("Y-m-d H:i:s"),
                    'read_status' => 0
                );
                Notification::insert($notificationPost);
            }

            $tokens = array();
            $player_ids = array();
            $users = UserAuthLogin::where('user_id',$input['driver_id'])->get();
            if(count($users)>0){
                foreach ($users as $row) {
                    if(setting('is_enable_one_signal')){
                        if($row->player_id!=null){
                            array_push($player_ids,$row->player_id);
                        }
                    }else{
                        if($row->fcm_token!=null){
                            array_push($tokens,$row->fcm_token);
                        }
                    }
                }
            }
            if(setting('is_enable_one_signal')){
                \Helpers::sendNotification($player_ids,"Order Assigned successfully.","Orders Assigned",$key);
            }else{
                \Helpers::sendNotification($tokens,"Order Assigned successfully.","Orders Assigned",$key);
            }


        } catch (ValidatorException $e) {
            return $this->sendError($e->getMessage());
        }

        return $this->sendResponse([],__('lang.message_status_updated_successfully'));
    }

    public function assignDriverForDeliver(Request $request)
    {
        $input = $request->all();
        $id = $input['order_item_id'];
        try {
            $orderPost = array(
                'driver_id' => $input['driver_id']
            );
            $order_id = 0;
            $orderProduct = OrderProductsDelivery::where('id',$id)->first();

            if($orderProduct){
                $order_id = $orderProduct->order_id;
            }
            OrderProductsDelivery::where(['order_id'=>$orderProduct->order_id,'product_id'=>$orderProduct->product_id])->update($orderPost);

            $key = setting('fcm_key');

            if($input['driver_id'] != ''){
                $notificationPost = array(
                    'user_id' => $input['driver_id'],
                    'order_id' => $orderProduct->order_id,
                    'order_item_id' => $id,
                    'lang_code' => 'en',
                    'title' => "Order Assigned",
                    'message' => "Order Assigned successfully.",
                    'created_at' => date("Y-m-d H:i:s"),
                    'read_status' => 0
                );
                Notification::insert($notificationPost);
            }else{
                Notification::where('order_item_id',$id)->delete();
            }

            $tokens = array();
            $player_ids = array();
            $users = UserAuthLogin::where('user_id',$input['driver_id'])->get();
            if(count($users)>0){
                foreach ($users as $row) {
                    if(setting('is_enable_one_signal')){
                        if($row->player_id!=null){
                            array_push($player_ids,$row->player_id);
                        }
                    }else{
                        if($row->fcm_token!=null){
                            array_push($tokens,$row->fcm_token);
                        }
                    }
                }
            }
            if(setting('is_enable_one_signal')){
                \Helpers::sendNotification($player_ids,"Order Assigned successfully.","Orders Assigned",$key,false,$order_id);
            }else{
                \Helpers::sendNotification($tokens,"Order Assigned successfully.","Orders Assigned",$key,false,$order_id);
            }
            
            return $this->sendResponse([],"Driver assign successfully.");


        } catch (ValidatorException $e) {
            return $this->sendError($e->getMessage());
        }

        return $this->sendResponse([],__('lang.message_driver_assign_successfully'));
    }

    /**
     * Display a listing of the revenue.
     *
     * @return \Illuminate\Http\Response
     */
    public function revenue(Request $request)
    {
        $post = $request->all();
        $data = OrderProductsDelivery::getReport($post);

        $filter = array();
        $filter['user'] = User::where('id','!=',1)->where('type','user')->get();
        $filter['order_status'] = OrderStatus::get();
        $filter['drivers'] = User::where('type','driver')->where('status',1)->get();
        $filter['products'] = Product::where('status','=',1)->get();
        // Add the new payment method to the filters
        $filter['payment_methods'] = ['cash','razorpay', 'paypal', 'stripe'];
        // Add the new payment status to the filters
        $filter['payment_statuses'] = ['Paid', 'Processing','Waiting for Client', 'Failed'];
        return view('orders.revenue',compact('data'))->with('filter',$filter);
    }

    /**
     * Display a listing of the revenue.
     *
     * @return \Illuminate\Http\Response
     */
    public function activeSubscription(Request $request)
    {
        $post = $request->all();
        $data = OrderProduct::getActiveSubscrpition($post);
        $filter = array();
        $filter['user'] = User::where('id','!=',1)->where('type','user')->get();
        $filter['order_status'] = OrderStatus::get();
        $filter['drivers'] = User::where('type','driver')->where('status',1)->get();
        $filter['products'] = Product::where('status','=',1)->get();
        return view('orders.active_subscription',compact('data'))->with('filter',$filter);
    }

    public function export(Request $request)
    {
        // You can get sorting parameters from the request, or use default values
        $sortColumn = $request->input('sort_column', 'id'); // Default to 'id'
        $sortDirection = $request->input('sort_direction', 'asc'); // Default to 'asc'

        $fileName = 'Orders ' . date('d M Y h i s A') . '.xlsx';

        // Pass sorting parameters to OrdersExport
        return Excel::download(new OrdersExport($sortColumn, $sortDirection), $fileName, ExcelType::XLSX);
    }

    public function exportToPdf()
    {
        $currentDateTime = Date::now()->format('d M Y h i s A'); // Format: YearMonthDayHourMinuteSecond

        $data['orders'] =  Order::getExportDataToPdf();
        // return view('exports.orders_pdf',$data);
        $pdf = PDF::loadView('exports.orders_pdf', ['orders' => Order::getExportDataToPdf()])->setPaper('a4', 'landscape')->setOption('margin-right', 15)->setOption('margin-left', 15);


        $filename = 'Orders ' . $currentDateTime . '.pdf';

        return $pdf->download($filename);
    }


    public function exportRevenueReportToExcel(Request $request)
    {
         return Excel::download(new RevenueExport, 'Revenues ' . date("d M Y h i s A") . '.xlsx', ExcelType::XLSX);
    }



    public function exportRevenueReportToPdf()
    {
        $currentDateTime = Date::now()->format('d M Y h i s A'); // Format: YearMonthDayHourMinuteSecond

        $pdf = PDF::loadView('exports.revenue_pdf', ['orders' => OrderProductsDelivery::getRevenueReportToPdf()])->setPaper('a4', 'landscape')->setOption('margin-right', 15)->setOption('margin-left', 15);


        $filename = 'Revenues ' . $currentDateTime . '.pdf';

        return $pdf->download($filename);
    }

    public function cancelOrderRequest(Request $request)
    {
        $data = OrderCancelRequest::getLists();
        $filter = array();
        $filter['user'] = User::where('id','!=',1)->where('type','user')->get();
        $filter['order_status'] = OrderStatus::get();

        // Add the new payment method to the filters
        $filter['payment_methods'] = ['cash','online'];

        return view('orders.cancelrequest.index',compact('data'))->with('filter',$filter);
    }

    public function declinedOrderRequest($id)
    {
        OrderCancelRequest::where('id',$id)->update(array('status'=>2,'updated_at' => date("Y-m-d H:i:s")));
        return redirect()->back()->with('message',__('lang.message_status_updated_successfully'));
    }

    public function approvedOrderRequest($id)
    {
        $data = OrderCancelRequest::find($id);
        $postUpdate = array(
            'status'=>1,
            'updated_at' => date("Y-m-d H:i:s")
        );

        if($data->order){
            if($data->order->payment_method!='cash'){
                $totalWalletAmount = 0;
                $productsDelivery = OrderProductsDelivery::where('delivery_date','>=',date("Y-m-d"))->where('order_id',$data->order_id)->with('product_orders')->get();
                foreach ($productsDelivery as $row) {
                    if($row->product_orders){
                        $totalWalletAmount = $totalWalletAmount + $row->product_orders->per_delivery_amount;
                    }
                    OrderProductsDelivery::where('id',$row->id)->update(array('order_status_id'=> 4,'updated_at' => date("Y-m-d H:i:s")));
                    Order::where('id',$data->order_id)->update(array('order_status_id'=> 4,'updated_at' => date("Y-m-d H:i:s")));
                }
                $postUpdate['refund_amount'] = $totalWalletAmount;
                if($totalWalletAmount>0){
                    $postData = array(
                        'order_id'=>$data->order_id,
                        'user_id' => $data->user_id,
                        'description' => 'Your order has been cancelled and refund credited',
                        'type' => 'credit',
                        'amount' => $totalWalletAmount,
                        'created_at' => date("Y-m-d H:i:s")
                    );
                    UserWallet::insert($postData);
                    $postUpdate['refund_status'] = 1;
                }
            }else{
                $productsDelivery = OrderProductsDelivery::where('delivery_date','>=',date("Y-m-d"))->where('order_id',$data->order_id)->with('product_orders')->get();
                foreach ($productsDelivery as $row) {
                    OrderProductsDelivery::where('id',$row->id)->update(array('order_status_id'=> 4,'updated_at' => date("Y-m-d H:i:s")));
                    Order::where('id',$data->order_id)->update(array('order_status_id'=> 4,'updated_at' => date("Y-m-d H:i:s")));
                }
            }
        }

        OrderCancelRequest::where('id',$id)->update($postUpdate);

        return redirect()->back()->with('message',__('lang.message_status_updated_successfully'));
    }
}
