<?php

namespace App\Http\Controllers;
use anlutro\LaravelSettings\Facade as Setting;

use Illuminate\Http\Request;
use App\Models\LanguagesKeywords;
use App\Models\Languages;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;

class LanguagesKeywordsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $post = $request->all();
        $language = Languages::all();
        $data = LanguagesKeywords::getLists($post);

        // Step 1: Identify duplicates
        $duplicates = DB::table('keyword_translations')
            ->select('keyword', 'group', 'lang_code', DB::raw('COUNT(*) as count'))
            ->groupBy('keyword', 'group', 'lang_code')
            ->having('count', '>', 1)
            ->get();
    
        // Step 2: Delete duplicates
        foreach ($duplicates as $duplicate) {
            // Get the IDs of the duplicate records
            $idsToDelete = DB::table('keyword_translations')
                ->where('keyword', $duplicate->keyword)
                ->where('group', $duplicate->group)
                ->where('lang_code', $duplicate->lang_code)
                ->pluck('id') // Assuming 'id' is the primary key
                ->toArray();
    
            // Keep the first record and delete the rest
            array_shift($idsToDelete); // Remove the first ID to keep it
    
            if (!empty($idsToDelete)) {
                DB::table('keyword_translations')
                    ->whereIn('id', $idsToDelete)
                    ->delete();
            }
        }

        return view('keywords.index',compact('data'))->with('language',$language);
    }
    
    /**
     * Store the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'group' => 'required',
            'keyword' => 'required',
            'value' => 'required',
        ]);

        // Get the input data from the request
        $group = $request->input('group');
        $keyword = $request->input('keyword');
        $value = $request->input('value');

        // Set a default language code if not provided
        $langCode = $request->input('lang_code', 'en');

        // Check if the record already exists
        $existingTranslation = LanguagesKeywords::where('group', $group)
            ->where('keyword', $keyword)
            ->where('lang_code', $langCode)
            ->first();

        if ($existingTranslation) {

            return response()->json([
                'status' => false,
                'message' => __('lang.message_language_keyword_already_exist'),
            ]);
        } else {
            // Store the new record in the database
            LanguagesKeywords::create([
                'group' => $group,
                'keyword' => $keyword,
                'value' => $value,
                'lang_code' => $langCode,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Update language files as in the update function
            $this->updateLanguageFiles();

            // Redirect to the index page with a success message
            return response()->json([
                'status' => true,
                'message' => __('lang.message_record_created_successfully'),
            ]);
        }
    }


    // Helper function to update language files
    private function updateLanguageFiles()
    {
        $languages = Languages::where('status', 1)->get();
    
        //
        foreach ($languages as $lang) {
            $translations = LanguagesKeywords::whereIn('group', ['web', 'admin'])
                ->where('lang_code', $lang->language_code)
                ->get();
    
            $data = "<?php\nreturn [\n";
    
            foreach ($translations as $row) {
                $row->value = str_replace("'", "", $row->value);
                $data .= "    '" . $row->keyword . "' => '" . $row->value . "',\n";
            }
    
            $data .= "];";
    
            $fileName = 'lang.php';
            $path = resource_path("lang/{$lang->language_code}");
            $filePath = $path . '/' . $fileName;
    
            if (!File::exists($path)) {
                File::makeDirectory($path, 0755, true);
            }
    
            File::put($filePath, $data);
        }
    }


    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data = LanguagesKeywords::where('keyword',$id)->where('lang_code','en')->first();
        $languages = Languages::get();
        foreach ($languages as $row) {
            $row->details = LanguagesKeywords::where('keyword',$id)->where('lang_code',$row->language_code)->first();
            if(!$row->details){
                $postData = array(
                    'lang_code' => $row->language_code,
                    'group' => $data->group,
                    'keyword' => $data->keyword,
                    'value' => $data->value,
                    'created_at' => date("Y-m-d H:i:s")
                );
                LanguagesKeywords::insert($postData);
                $row->details = LanguagesKeywords::where('keyword',$id)->where('lang_code',$row->language_code)->first();
            }
        }
        return view('keywords.edit',compact('languages'))->with('id',$id);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {

        request()->validate([
            'language_code' => 'required',
            'keyword_value' => 'required',
        ]);

        $languageCodes = $request->input('language_code');
        $keywordValues = $request->input('keyword_value');
        $keyword = $request->input('keyword');

        for ($i = 0; $i < count($languageCodes); $i++) {

			$get = LanguagesKeywords::where(['lang_code' => $languageCodes[$i], 'keyword' => $keyword])->first();

            if($get){
				LanguagesKeywords::where('id',$get->id)->update(['value' => $keywordValues[$i]]);
				$getApi = LanguagesKeywords::where(['lang_code' => $languageCodes[$i], 'keyword' => $keyword,'group' => 'api'])->first();
				if ($getApi) {
                    $getApi->update(['value' => $keywordValues[$i]]);
                } else {
                    LanguagesKeywords::create([
                        'lang_code' => $languageCodes[$i],
                        'keyword' => $keyword,
                        'value' => $keywordValues[$i],
                        'group' => 'api',
                    ]);
                }
            }
        }

        $languages = Languages::where('status',1)->get();
        foreach ($languages as $lang) {
            $translations = LanguagesKeywords::where('lang_code',$lang->language_code)->get();
            $data = "<?php\nreturn [\n";

            foreach ($translations as $row) {
                $row->value = str_replace("'", "", $row->value);
                $data .= "    '" . $row->keyword . "' => '" . $row->value . "',\n";
            }

            $data .= "];";

            $fileName = 'lang.php';
            $path = resource_path("lang/{$lang->language_code}");
            $filePath = $path . '/' . $fileName;

            if (!File::exists($path)) {
                File::makeDirectory($path, 0755, true);
            }

            File::put($filePath, $data);
        }

        return redirect()->route('keywords.index')
                        ->with('success',__('lang.message_record_updated_successfully'));
    }

}
