<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\DeliveryAddresses;
use App\Models\AppSetting;
use App\Models\UserAuthLogin;
use App\Models\Languages;
use App\Models\Timeslots;
use App\Models\TimeslotsTranslation;
use App\Models\CmsPages;
use App\Models\CmsPagesTranslation;
use App\Models\Partners;
use App\Models\PartnersTranslation;
use App\Models\NonWorkingDays;
use App\Models\DeliveryAreas;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use App\Http\Traits\TwilioTrait;
use Illuminate\Support\Facades\Log;


class UserAPIController extends Controller
{

    use TwilioTrait;
    private $language;

    public function __construct(Request $request)
    {
        // parent::__construct();
        $this->request = $request;
        $this->language = $request->header('lang-code') && $request->header('lang-code') != '' ? $request->header('lang-code') : 'en';
        app()->setLocale($this->language);
    }

    public function doLogin(Request $request)
    {
        $this->validate($request, [
            'phone' => 'required'
        ]);

        $phoneInput = $request->input('phone');

        // Use regex to separate country code and phone number
        preg_match('/^(\+\d+)\s*(\d+)$/', $phoneInput, $matches);

        $country_code = $matches[1] ?? null;
        $phone = $matches[2] ?? null;
    
        $phone = preg_replace('/\s+/', '', $phoneInput);
        
        if($request->has('otp') && $request->input('otp') !=''){
           $otp = $request->input('otp'); 
        }else{
            
            if(setting('enable_test_mode') == 1){
                $otp = 123456;
            }else{
                $otp = rand(999999,100000);
            }
            
        }
        
        $testMode = setting('enable_test_mode', false);
        $twilioEnabled = setting('is_enable_twilio');
        $fast2SmsEnabled = setting('is_enabled_fast_2_sms');
        $Factor2SmsEnabled = setting('is_enabled_2_factor');
    
        // Include soft-deleted records in the query
        $user = User::withTrashed()->where('phone', $phone)->first();
    
        if ($user) {
            if ($user->trashed()) {
                $user->restore();
            }
    
            $postUpdate = [
                'country_code' => $country_code,
                'is_verified' => 0,
                'otp' => $otp
            ];
            User::where('id', $user->id)->update($postUpdate);
    
            if ($twilioEnabled && !$testMode) {
                $status_code = \Helpers::sendSms($otp, $phone);
                if($status_code != 201 ){
                    return $this->sendError(__('lang.message_something_went_wrong'), 500);
                }
            }


            if ($Factor2SmsEnabled == 1) {
                $status_code = \Helpers::send2FactorSms($otp, $phone);
                if($status_code != 201 ){
                    return $this->sendError(__('lang.message_something_went_wrong'), 500);
                }
            }
    
            $response = [
                'id' => $user->id,
                'phone' => $user->phone,
                'otp' => $otp,
                'is_verified' => 0
            ];
    
            return $this->sendResponse($response, __('lang.message_user_login_successfully_proceed_to_otp_verification'));
        } else {
    
            $postInsert = [
                'phone' => $phone,
                'is_verified' => 0,
                'otp' => $otp,
                'password' => Hash::make("123456"),
                'country_code' => $country_code,
                'created_at' => now()
            ];
            $id = User::insertGetId($postInsert);
    
            if ($twilioEnabled && !$testMode) {
                $status_code = \Helpers::sendSms($otp, $phone);
                if($status_code != 201){
                    return $this->sendError(__('lang.message_something_went_wrong'), 500);
                }
            }
            
            if ($fast2SmsEnabled == 1) {
                \Helpers::sendFast2Sms($otp, $phone);
            }
    
            $response = [
                'id' => $id,
                'phone' => $phone,
                'otp' => $otp,
                'is_verified' => 0
            ];
    
            return $this->sendResponse($response, 'User Login successfully. Proceed to OTP verification.');
        }
    }

    
    public function doVerifyOtp(Request $request)
    {
        // Get and sanitize phone number by removing any spaces
        $phone = preg_replace('/\s+/', '', $request->input('phone'));
    
        $user = User::where('phone', $phone)->first();
    
        if ($user) {
            // Check if OTP matches or is the test OTP
            if ($user->otp) {
                $inputOtp = $request->input('otp'); // Declare OTP input variable
    
                if (setting('is_enable_twilio') == 1 || setting('is_enabled_fast_2_sms')) {
                    $checkOtp = $user->otp;
    
                    if ($inputOtp == $checkOtp) {
    
                        $user->is_verified = 1;
                        $user->save();
    
                        $postUpdate = [
                            'user_id' => $user->id,
                            'api_token' => \Helpers::generateApiToken(),
                            'fcm_token' => $request->input('fcm_token'),
                            'player_id' => $request->input('player_id'),
                            'device_type' => $request->input('device_type'),
                            'created_at' => now()
                        ];
                        UserAuthLogin::insert($postUpdate);
    
                        return $this->sendResponse($postUpdate, __('lang.message_user_verified_logged_in_successfully'));
                    } else {
                        return $this->sendError(__('lang.message_invalid_otp'), 401);
                    }
                } else {
                    
                    if($inputOtp == '123456'){
        
                        $user->is_verified = 1;
                        $user->save();
        
                        $postUpdate = [
                            'user_id' => $user->id,
                            'api_token' => \Helpers::generateApiToken(),
                            'fcm_token' => $request->input('fcm_token'),
                            'player_id' => $request->input('player_id'),
                            'device_type' => $request->input('device_type'),
                            'created_at' => now()
                        ];
                        UserAuthLogin::insert($postUpdate);
        
                        return $this->sendResponse($postUpdate, __('lang.message_user_verified_logged_in_successfully'));
                    }else{
                        return $this->sendError(__('lang.message_invalid_otp'), 401);  
                    }
                }
                
                return $this->sendError(__('lang.message_invalid_otp'), 401);
    
            } else if (setting('is_enable_firebase')) {
                $user->is_verified = 1;
                $user->save();
    
                $postUpdate = [
                    'user_id' => $user->id,
                    'api_token' => \Helpers::generateApiToken(),
                    'fcm_token' => $request->input('fcm_token'),
                    'player_id' => $request->input('player_id'),
                    'device_type' => $request->input('device_type'),
                    'created_at' => now()
                ];
                UserAuthLogin::insert($postUpdate);
                return $this->sendResponse($postUpdate, __('lang.message_user_verified_logged_in_successfully'));
            } else {
                return $this->sendError(__('lang.message_invalid_otp'), 401);
            }
        } else {
            return $this->sendError(__('lang.message_user_not_found'), 404);
        }
    }

    
    public function doResndOtp(Request $request)
    {
        $this->validate($request, [
            'phone' => 'required'
        ]);
    
        $user = User::where('phone', $request->input('phone'))->first();
    
        if ($user) {
            $response = [
                'id' => $user->id,
                'phone' => $user->phone,
                'is_verified' => 0
            ];
    
            if (!setting('is_enable_firebase')) {
                \Helpers::sendSms('Your One Time Password is ' . $user->otp, $request->input('phone'));
            }
    
            return $this->sendResponse($response, __('lang.message_otp_send_successfully'));
        } else {
            return $this->sendError(__('lang.message_user_not_found'), 401);
        }
    }

    function doDriverLogin(Request $request)
    {
        $this->validate($request, [
            'phone' => 'required',
            'password' => 'required',
        ]);
        $user = User::where(['phone' => $request->phone])->first();
        if ($user) {
            if (auth()->attempt(['phone' => $request->input('phone'), 'password' => $request->input('password')])) {
                if ($user->status == 1) {
                    $postUpdate = array(
                        'user_id' => $user->id,
                        'api_token' => \Helpers::generateApiToken(),
                        'player_id' => $request->input('player_id'),
                        'fcm_token' => $request->input('fcm_token'),
                        'fcm_token' => $request->input('fcm_token'),
                        'device_type' => $request->input('device_type'),
                        'created_at' => date("y-m-d H:i:s")
                    );
                    UserAuthLogin::insert($postUpdate);
                    return $this->sendResponse($postUpdate,__('lang.message_data_retrieve_successfully'));
                } else {
                    return $this->sendError(__('lang.message_driver_is_inactive_now'));
                }
            }else{
                return $this->sendError(__('lang.message_invalid_password'));
            }
        } else {
            return $this->sendError(__('lang.message_invalid_phone'));
        }
    }

    function getProfile(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    $api_token = '';
                    if($request->header('api-token')!=''){
                        $api_token = $request->header('api-token');
                    }
                    
                    $user = $request->userAuthData->user;
                    $fcmToken = $request->userAuthData['fcm_token'] ?? null;
                    $user->fcmToken = $fcmToken;
                    $user->api_token = $api_token;
                    return $this->sendResponse($user, __('lang.message_data_retrieve_successfully'));
                }
            }
            return $this->sendError(__('lang.message_user_not_found'), 401);
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    function getDriverProfile(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    return $this->sendResponse($request->userAuthData->user, __('lang.message_data_retrieve_successfully'));
                }
            }
            return $this->sendError(__('lang.message_user_not_found'), 401);
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    public function doUpdateProfile(Request $request)
    {
        try {
            // Validate input data
            $request->validate([
                'name' => 'sometimes|string|max:255',
                'email' => 'sometimes|email|max:255',
                'phone' => 'sometimes|string|max:20',
                'lang_code' => 'sometimes|string|max:5',
            ]);

            // Check if user data exists
            if (!$request->userAuthData || !$request->userAuthData->user) {
                return $this->sendError(__('lang.message_user_not_found'), 401);
            }

            $post = $request->all();
            $user = $request->userAuthData->user;

            // Check for existing phone number
            if ($request->input('phone')) {
                $userPhone = User::where('phone', $request->input('phone'))->where('id', '!=', $user->id)->first();
                if ($userPhone) {
                    return $this->sendError(__('lang.message_mobile_number_already_exists'), 422);
                }
            }

            // Check for existing email
            if ($request->input('email')) {
                $userEmail = User::where('email', $request->input('email'))->where('id', '!=', $user->id)->first();
                if ($userEmail) {
                    return $this->sendError(__('lang.message_email_already_exists'), 422);
                }
            }

            // Prepare update data
            $postData = array_filter([
                'name' => $post['name'] ?? null,
                'language_code' => $post['lang_code'] ?? null,
                'phone' => $post['phone'] ?? null,
                'email' => $post['email'] ?? null,
            ]);

            // Update user data
            User::where('id', $user->id)->update($postData);
            $user->refresh(); // Refresh the model instance with updated data

            return $this->sendResponse($user, __('lang.message_data_updated_successfully'));
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage(), 500);
        }
    }

    public function doUpdateDriverProfile(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    $post = $request->all();
                    $user = $request->userAuthData->user;

                    $user2 = User::where('phone', $request->input('phone'))->where('id', '!=', $user->id)->first();
                    if ($user2) {
                        return $this->sendError(__('lang.message_mobile_number_already_exists'), 401);
                    }

                    $user3 = User::where('email', $request->input('email'))->where('id', '!=', $user->id)->first();
                    if ($user3) {
                        return $this->sendError(__('lang.message_email_already_exists'), 401);
                    }

                    $postData = array(
                        'updated_at' => date("Y-m-d H:i:s")
                    );
                    if (isset($post['name'])) {
                        $postData['name'] = $post['name'];
                    }

                    if (isset($post['lang_code'])) {
                        $postData['language_code'] = $post['lang_code'];
                    }

                    if (isset($post['phone'])) {
                        $postData['phone'] = $post['phone'];
                    }

                    if (isset($post['email'])) {
                        $postData['email'] = $post['email'];
                    }
                    User::where('id', $user->id)->update($postData);
                    $user = User::where('id', $user->id)->first();
                    return $this->sendResponse($user, __('lang.message_profile_updated_successfully'));
                }
            }
            return $this->sendError('User not found', 401);
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }

    public function doUpdateNotification(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    $post = $request->all();
                    $user = $request->userAuthData->user;
                    $postData = array(
                        'updated_at' => date("Y-m-d H:i:s")
                    );

                    if (isset($post['enable_notificaton'])) {
                        $postData['enable_notificaton'] = $post['enable_notificaton'];
                    } else {
                        $postData['enable_notificaton'] = 0;
                    }
                    User::where('id', $user->id)->update($postData);
                    $user = User::where('id', $user->id)->first();
                    return $this->sendResponse($user, __('lang.message_data_updated_successfully'));
                }
            }
            return $this->sendError(__('lang.message_user_not_found'), 401);
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }

    public function listUserAddress(Request $request)
    {
        try {
            $user_id = 0;
            if ($request->userAuthData) {
                $user_id = $request->userAuthData->user_id;
            }
            $results = DeliveryAddresses::where('user_id', $user_id)->get();
            return $this->sendResponse($results, __('lang.message_data_retrieve_successfully'));
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    public function addUpdateUserAddress(Request $request)
    {
        try {

            $user_id = 0;
            if ($request->userAuthData) {
                $user_id = $request->userAuthData->user_id;
            }
            $post = $request->all();
            $post['user_id'] = $user_id;

            if ($post['id']) {
                DeliveryAddresses::where('id', $post['id'])->update($post);
            } else {
                DeliveryAddresses::create($post);
            }
            $results = DeliveryAddresses::where('user_id', $user_id)->get();
            return $this->sendResponse($results, __('lang.message_data_created_successfully'));
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    public function deleteUserAddress($id, Request $request)
    {
        try {

            DeliveryAddresses::where('id', $id)->delete();
            return $this->sendResponse([], __('lang.message_data_deleted_successfully'));
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    public function setDefaultUserAddress($id, Request $request)
    {
        try {
            $user_id = 0;
            if ($request->userAuthData) {
                $user_id = $request->userAuthData->user_id;
            }
            $post = $request->all();
            $results = DeliveryAddresses::where('user_id', $user_id)->update(array('is_default' => $post['is_default']));
            DeliveryAddresses::where('id', $id)->update(array('is_default' => 1));

            return $this->sendResponse([], __('lang.message_data_updated_successfully'));
        } catch (RepositoryException $e) {
            return $this->sendError($e->getMessage());
        }
    }

    function settings(Request $request)
    {
        $response = array();
        $response['delivery_areas'] = DeliveryAreas::where('status', 1)->with(['state', 'cities'])->get();
        foreach($response['delivery_areas'] as $row){
            $row->state_name = $row->state->state_name ?? null;
            $row->city_name = $row->cities->city_name ?? null;
            unset($row->state);
            unset($row->cities);
        }

        $response['holidays'] = NonWorkingDays::where('status', 1)->get();
        $response['setting'] = AppSetting::all();
        foreach($response['setting'] as $row){
            if(isset($row) && $row->key == 'splash_screen'){
                $row->value = asset('site-assets/images/app/'.$row->value);
            }
            
        }
        $languages = Languages::all()->map(function($language) {
            // Check if is_default is an integer 1, and convert it to string "1"
            $language->is_default = (string)$language->is_default;
            return $language;
        });
        $response['languages'] = $languages;
        $response['timeslots'] = Timeslots::where('status',1)->get();
        $response['terms'] = CmsPages::where('page_name', 'terms')->first();
        if ($response['terms']) {
            $translated_data = CmsPagesTranslation::where('lang_code', $this->language)->where('cms_id', $response['terms']->id)->first();
            if ($translated_data) {
                if ($translated_data->page_title != '') {
                    $response['terms']->page_title = $translated_data->page_title;
                }
                if ($translated_data->description != '') {
                    $response['terms']->description = $translated_data->description;
                }
            }
        }

        $response['about_us'] = CmsPages::where('page_name', 'about')->get();
        foreach ($response['about_us'] as $row) {
            $translated_data = CmsPagesTranslation::where('lang_code', $this->language)->where('cms_id', $row->id)->first();
            if ($translated_data) {
                if ($translated_data->page_title != '') {
                    $row->page_title = $translated_data->page_title;
                }
                if ($translated_data->description != '') {
                    $row->description = $translated_data->description;
                }
            }
        }
        
        
        $response['privacy_policy'] = CmsPages::where('page_name', 'policy')->get();
        foreach ($response['privacy_policy'] as $row) {
            $translated_data = CmsPagesTranslation::where('lang_code', $this->language)->where('cms_id', $row->id)->first();
            if ($translated_data) {
                if ($translated_data->page_title != '') {
                    $row->page_title = $translated_data->page_title;
                }
                if ($translated_data->description != '') {
                    $row->description = $translated_data->description;
                }
            }
        }


        foreach ($response['timeslots'] as $row) {
            $translated_data = TimeslotsTranslation::where('lang_code', $this->language)->where('timeslot_id', $row->id)->first();
            if ($translated_data) {
                if ($translated_data->name != '') {
                    $row->name = $translated_data->name;
                }
            }
        }

        $response['partners'] = Partners::where('type', 'partner')->get();
        foreach ($response['partners'] as $row) {
            $translated_data = PartnersTranslation::where('lang_code', $this->language)->where('partner_id', $row->id)->first();
            if ($translated_data) {
                if ($translated_data->name != '') {
                    $row->name = $translated_data->name;
                }
            }
        }

        $response['awards'] = Partners::where('type', 'awards')->get();
        foreach ($response['awards'] as $row) {
            $translated_data = PartnersTranslation::where('lang_code', $this->language)->where('partner_id', $row->id)->first();
            if ($translated_data) {
                if ($translated_data->name != '') {
                    $row->name = $translated_data->name;
                }
            }
        }

        return $this->sendResponse($response, __('lang.message_data_retrieve_successfully'));
    }

    public function doLogout(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    $post = $request->all();
                    $user = $request->userAuthData->user;
                    $postUpdate = array(
                        'user_id' => $user->id,
                        'api_token' => '',
                        'player_id' => '',
                        'fcm_token' => '',
                        'updated_at' => date("y-m-d H:i:s")
                    );
                    UserAuthLogin::where('id', $request->userAuthData->id)->update($postUpdate);
                    return $this->sendResponse([], __('lang.message_user_logout_successfully'));
                }
            }
            return $this->sendError(__('lang.message_user_not_found'), 401);
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }

    public function doDriverLogout(Request $request)
    {
        try {
            if ($request->userAuthData) {
                if ($request->userAuthData->user) {
                    $post = $request->all();
                    $user = $request->userAuthData->user;
                    $postUpdate = array(
                        'user_id' => $user->id,
                        'api_token' => '',
                        'player_id' => '',
                        'fcm_token' => '',
                        'updated_at' => date("y-m-d H:i:s")
                    );
                    UserAuthLogin::where('id', $request->userAuthData->id)->update($postUpdate);
                    return $this->sendResponse([], __('lang.message_driver_logout_successfully'));
                }
            }
            return $this->sendError(__('lang.message_driver_not_found'), 401);
        } catch (\Exception $ex) {
            return $this->sendError($ex->getMessage());
        }
    }

    public function sendOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phone' => 'required'
            ]);
            $phoneNumber = $request->input('phone');
            $twillo_service_id=$this->twilio_create_services();
            $verification = $this->twilio_create_sms_otp( $twillo_service_id,$phoneNumber);
            return response()->json(['success' => true, 'service_id'=> $twillo_service_id, 'message' => __('lang.message_otp_send_successfully')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_failed_to_send_otp')]);
        }
    }
    
    public function verifyOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phonenumber' => 'required',
                'otp' => 'required',
                'service_id' => 'required'
            ]);
            $response = $this->twilio_verify_otp( $request->service_id,$request->phonenumber,$request->otp);

            if(empty($response)){
                $response = [
                    'success' => false,
                    'message' => __('lang.message_invalid_otp'),
                ];
                return response()->json($response, 400);
            }
            if(is_string($response)){
                $response = [
                    'success' => false,
                    'message' => __('lang.message_invalid_otp'),
                ];
                return response()->json($response, 400);
            }
            if($response->status!='approved'){
                $response = [
                    'success' => false,
                    'message' => __('lang.message_invalid_otp'),
                ];
                return response()->json($response, 400);
            }
            return response()->json(['success' => true, 'verified'=> true, 'message' => __('lang.message_otp_send_successfully')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_failed_to_send_otp')]);
        }
    }
    
    
    public function sendFast2SmsOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phone' => 'required'
            ]);
            $phoneNumber = $request->input('phone');
            $otp =  rand(100000, 999999);
            if($phoneNumber){
                \Helpers::sendFast2Sms($otp , $phoneNumber);
            }
            
            
            $user = User::where('phone', $phoneNumber)->first();
            if($user){
                
                // Update user's OTP and verification status
                $postUpdate = [
                    'is_verified' => 0,
                    'otp' => $otp
                ];
                User::where('id', $user->id)->update($postUpdate);
            
            }else{
                 
                // Create new user and set OTP and password
                $postInsert = [
                    'phone' => $phone,
                    'is_verified' => 0,
                    'otp' => $otp,
                    'password' => Hash::make("123456"),
                    'created_at' => now()
                ];
                $id = User::insertGetId($postInsert);
            }
            
            return response()->json(['success' => true, 'service_id'=> $twillo_service_id, 'message' =>  __('lang.message_otp_send_successfully')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_failed_to_send_otp')]);
        }
    }
    
    public function verifyFast2SmsOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phonenumber' => 'required|regex:/^\+\d{8,15}$/',
                'otp' => 'required|numeric'
            ]);
    
            $phoneNumber = $request->input('phonenumber');
            $otp = $request->input('otp');
    
            // Mock verification for demo
            $user = User::where('phone', $phoneNumber)->where('otp', $otp)->first();
    
            if ($user) {
                $user->update(['is_verified' => 1]);
    
                return response()->json(['success' => true, 'verified' => true, 'message' => __('lang.message_otp_verified_successfully')]);
            }
    
            return response()->json(['success' => false, 'verified' => false, 'message' =>  __('lang.message_otp_verification_failed')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_otp_verification_failed')]);
        }
    }
    
    public function send2FactorOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phone' => 'required'
            ]);
            $phoneNumber = $request->input('phone');
            $otp =  rand(100000, 999999);
            if($phoneNumber){
                \Helpers::send2FactorSms($otp , $phoneNumber);
            }
            
            
            $user = User::where('phone', $phoneNumber)->first();
            if($user){
                
                // Update user's OTP and verification status
                $postUpdate = [
                    'is_verified' => 0,
                    'otp' => $otp
                ];
                User::where('id', $user->id)->update($postUpdate);
            
            }else{
                 
                // Create new user and set OTP and password
                $postInsert = [
                    'phone' => $phone,
                    'is_verified' => 0,
                    'otp' => $otp,
                    'password' => Hash::make("123456"),
                    'created_at' => now()
                ];
                $id = User::insertGetId($postInsert);
            }
            
            return response()->json(['success' => true, 'service_id'=> $twillo_service_id, 'message' => __('lang.message_otp_send_successfully')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_failed_to_send_otp')]);
        }
    }
    
    public function verify2FactorOtp(Request $request)
    {
        try {
            $this->validate($request, [
                'phonenumber' => 'required|regex:/^\+\d{8,15}$/',
                'otp' => 'required|numeric'
            ]);
    
            $phoneNumber = $request->input('phonenumber');
            $otp = $request->input('otp');
    
            // Mock verification for demo
            $user = User::where('phone', $phoneNumber)->where('otp', $otp)->first();
    
            if ($user) {
                $user->update(['is_verified' => 1]);
    
                return response()->json(['success' => true, 'verified' => true, 'message' => __('lang.message_otp_verified_successfully')]);
            }
    
            return response()->json(['success' => false, 'verified' => false, 'message' => __('lang.message_otp_verification_failed')]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => __('lang.message_otp_verification_failed')]);
        }
    }
    
}