<?php

namespace App\Http\Controllers\API;
use anlutro\LaravelSettings\Facade as Setting;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AppSetting;
use App\Models\Order;
use App\Models\Cart;
use App\Models\CardDetails;
use App\Models\UserPayment;
use App\Models\User;
use App\Models\ProductTranslation;
use App\Models\Product;
use App\Models\OrderProduct;
use App\Models\OrderProductsDelivery;
use App\Models\UserWallet;
use App\Models\UserAuthLogin;
use App\Models\OrderCancelRequest;
use App\Models\Notification;
use DB;
use Illuminate\Support\Facades\Mail;
use App\Notifications\FcmNotification;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class OrdersAPIController extends Controller
{
    private $language;
    protected $num_per_page = '10';
    public function __construct(Request $request)
    {
        parent::__construct();
        $this->request = $request;
        $this->language = $request->header('lang-code') && $request->header('lang-code') != '' ? $request->header('lang-code') : 'en';
        app()->setLocale($this->language);
    }

    public function list(Request $request)
    {
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }
        $orders = Order::where('user_id',$user_id)->with('product_orders')->with('order_status')->with('cancel_request')->orderBy('id','DESC')->get();

        foreach($orders as $row){
            foreach($row->product_orders as $prow){
                if($prow->product){
                    $translated_data = ProductTranslation::where('lang_code',$this->language)->where('product_id',$prow->product_id)->first();
                    if($translated_data){
                        if($translated_data->name!=''){
                            $prow->product->name = $translated_data->name;
                        }
                        if($translated_data->description!=''){
                            $prow->product->description = $translated_data->description;
                        }
                    }
                }
            }
        }
        return $this->sendResponse($orders, __('lang.admin_orders_retrieved_successfully'));
    }

    public function details($id,Request $request)
    {
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }
        $orders = $this->getOrderDetails($id,$user_id);

        if($orders){
            $orders->create_at = $orders->created_at
                ? \Helpers::commonDateTimeFormate($orders->created_at)
                : '--';
        }
        return $this->sendResponse($orders,__('lang.admin_orders_retrieved_successfully') );
    }

    private function getOrderDetails($id,$user_id)
    {
        $orders = Order::where('user_id',$user_id)->where('id',$id)->with('product_orders')->with('order_status')->with('payment')->with('delivery_address')->with('cancel_request')->first();
        if($orders){
            foreach($orders->product_orders as $prow){
                if($prow->product){
                    $translated_data = ProductTranslation::where('lang_code',$this->language)->where('product_id',$prow->product_id)->first();
                    if($translated_data){
                        if($translated_data->name!=''){
                            $prow->product->name = $translated_data->name;
                        }
                        if($translated_data->description!=''){
                            $prow->product->description = $translated_data->description;
                        }
                    }
                }
            }
        }
        return $orders;
    }

    public function placeOrder(Request $request)
    {
        $payment = $request->only('payment');
        if (isset($payment['payment']) && $payment['payment']['method']) {
            if($payment['payment']['method'] == 'cash' || $payment['payment']['method'] == 'wallet'){
                return $this->cashPayment($request);
            } else{
                return $this->cashPayment($request);
            }
        }
    }


    private function cashPayment(Request $request)
    {
        $input = $request->all();
        $user_id = $request->userAuthData->user_id ?? 0;
        // Check if the user exists
        if (!$user_id) {
            return $this->sendError('User not found', 401);
        }

        try {
            // Handle wallet payment method
            if ($input['payment']['method'] == 'wallet') {
                $total_wallet = UserWallet::getWalletbalance($user_id);
                if ($total_wallet < $input['final_amount']) {
                    return $this->sendError(__('lang.message_do_hot_have_sufficient_blance_in_you_wallet'), 401);
                }
            }
            // Prepare order data
            $orderPostData = $request->only([
                'timeslot_id', 'order_status_id', 'tax', 'delivery_address_id',
                'delivery_fee', 'is_coupon_applied', 'coupon_code',
                'promotional_disount', 'subtotal', 'final_amount'
            ]);
            $orderPostData['delivery_fee'] = 
                $orderPostData['subtotal'] < setting('minimum_order_value') 
                ? setting('shipping_charge') 
                : 0;
            $orderPostData['user_id'] = $user_id;
            $orderPostData['payment_method'] = $input['payment']['method'];
            $orderPostData['created_at'] = now();  // Using Laravel's now() helper

            // Insert the order and get the order ID
            $orderId = Order::insertGetId($orderPostData);

            // Process cart items and calculate order amount
            $cartData = Cart::where('user_id', $user_id)->with('product')->get();
            $paymentDetails = $this->extractPaymentDetails($input);
            $order_status_id = $this->determineOrderStatus($input, $orderPostData['payment_method'], $paymentDetails['payment_status']);
            
            if($paymentDetails['payment_status'] == 1){
                $this->processCartItems($cartData, $user_id, $orderId, $order_status_id);
            }

            // Handle payment and update order information
            $paymentId = $this->savePaymentDetails($orderPostData, $paymentDetails);
            $this->updateOrderPayment($orderId, $paymentDetails, $paymentId, $order_status_id);

            // If the payment method is wallet, deduct the amount from the wallet
            if ($input['payment']['method'] == 'wallet') {
               $this->deductFromWallet($user_id, $orderId, $input['final_amount']);
            }

            // Get order details for the response
            $order = $this->getOrderDetails($orderId, $user_id);
            // Handle post-payment actions
            if ($paymentDetails['payment_status'] == 1) { // Payment successful
                Cart::where('user_id', $user_id)->delete(); // Clear the user's cart

                // Handle notifications
                $this->sendNotifications($user_id, $orderId, $order);

                // Handle email notifications
                $this->sendOrderEmails($user_id, $order);
                return $this->sendResponse($order, __('lang.saved_successfully', ['operator' => __('Order')]));
            }else if($input['payment']['method'] == 'cash' || $input['payment']['method'] == 'wallet') {
                Cart::where('user_id', $user_id)->delete(); // Clear the user's cart

                // Handle notifications
                $this->sendNotifications($user_id, $orderId, $order);

                // Handle email notifications
                $this->sendOrderEmails($user_id, $order);
                
                return $this->sendResponse($order, __('lang.saved_successfully', ['operator' => __('Order')]));
            }else{
                // Payment failed
                return $this->sendResponse($order, __('lang.payment_failed', ['operator' => __('Order')]));
            }
            return $this->sendResponse($order, __('lang.saved_successfully', ['operator' => __('Order')]));
        } catch (ValidatorException $e) {
            return $this->sendError($e->getMessage());
        }
        
    }

    public function orderCancelRequest($id,Request $request)
    {
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }
        $orders = $this->getOrderDetails($id,$user_id);


        $postData = array(
            'order_id' => $id,
            'user_id' => $user_id,
            'refund_amount' => 0,
            'refund_status' => 0,
            'status' => 0,
            'created_at' => date("Y-m-d H:i:s"),
            'updated_at' => date("Y-m-d H:i:s")
        );


        $data = Order::find($id);
        if($data->payment_method!='cash' || $data->payment_method!='wallet'){
            $totalWalletAmount = 0;
            $productsDelivery = OrderProductsDelivery::where('order_status_id','<',4)->where('delivery_date','>=',date("Y-m-d"))->where('order_id',$id)->with('product_orders')->get();
            foreach ($productsDelivery as $row) {
                if($row->product_orders){
                    $totalWalletAmount = $totalWalletAmount + $row->product_orders->per_delivery_amount;
                }
            }
            $postData['refund_amount'] = $totalWalletAmount;
        }
        $postData['refund_status'] = 4;
        OrderCancelRequest::insertGetId($postData);

        return $this->sendResponse([], __('lang.alert_cancel_order_request_submited'));
    }

    public function getDriverAssignedOrderList(Request $request)
    {
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }
        $input = $request->all();
        $orderId = array();
        $order = [];
        $this->num_per_page = (isset($post['perpage']))?$post['perpage']:$this->num_per_page;
        if(isset($input['type']) && $input['type']=='past'){
            $productsDelivery = OrderProductsDelivery::whereIn('order_status_id', [4, 6, 7])
                ->where('driver_id', $user_id)
                ->with(['product_orders', 'delivery_status', 'user', 'order'])
                ->get();

            if(count($productsDelivery)){
                foreach($productsDelivery as $product){
                    if(!in_array($product->order_id,$orderId)){
                        array_push($orderId,$product->order_id);
                    }
                }
            }
            if(count($orderId)){
                $order = Order::whereIn('id',$orderId)->with('product_orders_driver')->with('order_status')->with('cancel_request')->with('userOnly')->with('delivery_address')->orderBy('id','DESC')->paginate($this->num_per_page)->appends('perpage', $this->num_per_page);
            }
        }
        else{
            $productsDelivery = OrderProductsDelivery::whereNotIn('order_status_id', [4, 5, 6, 7])
                ->where('driver_id', $user_id)
                ->where('delivery_date', date('Y-m-d'))
                ->with(['product_orders', 'delivery_status', 'user', 'order'])
                ->get();


            if(count($productsDelivery)){
                foreach($productsDelivery as $product){
                    if(!in_array($product->order_id,$orderId)){
                        array_push($orderId,$product->order_id);
                    }
                }
            }

            if(count($orderId) > 0){
                $order = Order::whereIn('id',$orderId)->with('product_orders_driver')->with('order_status')->with('cancel_request')->with('userOnly')->with('delivery_address')->orderBy('id','DESC')->paginate($this->num_per_page)->appends('perpage', $this->num_per_page);
            }
        }

        return $this->sendResponse($order, __('lang.message_data_retrieve_successfully'));
    }

    public function getDriverAssignedOrderDetail($id,Request $request)
    {
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }
        $orderId = array();
        $order = "";
        $productsDelivery = OrderProductsDelivery::where('driver_id', $user_id)
        ->where('order_id', $id)
        ->with(['product_orders', 'delivery_status', 'user', 'order'])
        ->first();

        if ($productsDelivery) {

            if (!in_array($productsDelivery->order_id, $orderId)) {
                array_push($orderId, $productsDelivery->order_id);
            }

        }
        if (count($orderId) > 0) {
            $order = Order::whereIn('id', $orderId)->with('product_orders_driver')->with('order_status')->with('cancel_request')->with('userOnly')->with('delivery_address')->orderBy('id', 'DESC')->first();
        }
        return $this->sendResponse($order, __('lang.message_data_retrieve_successfully'));    
        
    }

    public function changeOrderStatus($id,Request $request)
    {
        $input = $request->all();
        $user_id = 0;
        if($request->userAuthData){
            $user_id = $request->userAuthData->user_id;
        }

        OrderProductsDelivery::find($id)->update(['order_status_id'=>"$request->order_status_id"]);
        $order = OrderProductsDelivery::find($id);
        $orderId = ['0'=>$order->order_id];
        $productsDelivery = Order::whereIn('id', $orderId)->with('product_orders_driver')->with('order_status')->with('cancel_request')->with('userOnly')->with('delivery_address')->orderBy('id', 'DESC')->first();
        
        $orderData = Order::where('id',$order->order_id)->first();
        $oldOrder = OrderProductsDelivery::where('id',$id)->first();
        
        if($request->order_status_id == 7){
                if($orderData){
                    $payment_method = $orderData->payment_method;
                    if($payment_method == 'cash'){
                        
                        $opData = OrderProduct::where('id',$oldOrder->product_order_id)->first();
                    
                        $wallertData = [
                            'user_id' => $oldOrder->user_id,
                            'type' => 'credit',
                            'amount' => $opData->per_delivery_amount,
                            ];
                        
                        UserWallet::insert($wallertData);
                    }
                    
                }
            }

        return $this->sendResponse($productsDelivery, __('lang.message_status_change_successfully'));
    }

    function orderStatusList(Request $request)
    {
        $response = DB::table('order_statuses')
            ->whereIn('status', ['Pending', 'Order Received','Ready', 'On the Way', 'Delivered' , 'Not Delivered'])
            ->get();
    
        return $this->sendResponse($response, __('lang.message_data_retrieve_successfully'));
    }


    /* function which is used for getting and update details at the time of payment done in api */
    private function extractPaymentDetails($input)
    {
        if ($input['payment']['method'] == 'wallet') {
            $input['payment_status'] = 1;
        }

        if ($input['payment']['method'] == 'cash') {
            $input['payment_status'] = 1;
        }
            
        return [
            'payment_transection_id' => $input['payment_transection_id'] ?? 0,
            'payment_refferance_id' => $input['payment_refferance_id'] ?? 0,
            'payment_transection_data' => $input['payment_transection_data'] ?? 0,
            'payment_status' => $input['payment_status'] ?? 0,
            'status' => $this->manage_payment_status_field($input),
        ];
    }
    
    private function determineOrderStatus($input, $payment_method, $payment_status)
    {
        if(isset($input['payment_refferance_id'])){
            $input['payment_refferance_id'] = $input['payment_refferance_id'];
        }else{
          $input['payment_refferance_id']  = '';
        }
        
        if(isset($input['payment_transection_id'])){
            $input['payment_transection_id'] = $input['payment_transection_id'];
        }else{
          $input['payment_transection_id']  = '';
        }
        
        if ($input['payment_refferance_id'] == 0 || $input['payment_transection_id'] == 0 || $payment_status == 2 || $payment_status == 3) {
            return 1; // Failed/cancelled
        } elseif ($payment_status == 1) {
            return 1; // pending
        } elseif ($payment_method == 'cash' && $payment_status == 0) {
            return 1; // pending
        }elseif ($payment_method == 'wallet' && $payment_status == 0) {
            return 1; // pending
        }
        return 1;
    }

    private function processCartItems($cartData, $user_id, $orderId, $order_status_id)
    {
        foreach ($cartData as $cart) {
            if ($cart->product) {
                $orderItemData = [
                    'user_id' => $user_id,
                    'order_id' => $orderId,
                    'product_id' => $cart->product_id,
                    'quantity' => $cart->quantity,
                    'per_item_amount' => $cart->per_item_amount,
                    'per_delivery_amount' => $cart->per_delivery_amount,
                    'total_delivery_amount' => $cart->total_delivery_amount,
                    'no_of_delivery' => $cart->no_of_delivery,
                    'order_frequency' => $cart->order_frequency,
                    'start_date' => $cart->start_date,
                    'end_date' => $cart->end_date,
                    'days' => $cart->days,
                    'created_at' => now()
                ];
                $orderItemId = OrderProduct::insertGetId($orderItemData);

                $deliveryDates = explode(",", $cart->delivery_dates);
                foreach ($deliveryDates as $deliveryDate) {
                    $deliveryData = [
                        'user_id' => $user_id,
                        'order_id' => $orderId,
                        'product_id' => $cart->product_id,
                        'product_order_id' => $orderItemId,
                        'delivery_date' => date("Y-m-d", strtotime($deliveryDate)),
                        'order_status_id' => $order_status_id,
                        'created_at' => now()
                    ];
                    OrderProductsDelivery::insert($deliveryData);
                }
            }
        }
    }
    private function savePaymentDetails($orderPostData, $paymentDetails)
    {
        return UserPayment::insertGetId([
            "user_id" => $orderPostData['user_id'],
            "description" => 'Waiting for Client',
            "price" => $orderPostData['final_amount'],
            'payment_transection_id' => $paymentDetails['payment_transection_id'],
            'payment_refferance_id' => $paymentDetails['payment_refferance_id'],
            'payment_transection_data' => $paymentDetails['payment_transection_data'],
            'payment_status' => $paymentDetails['payment_status'],
            "status" => $paymentDetails['status'],
            "method" => $orderPostData['payment_method'],
            'created_at' => now()
        ]);
    }
    private function updateOrderPayment($orderId, $paymentDetails, $paymentId, $order_status_id)
    {
        Order::where('id', $orderId)->update([
            'payment_id' => $paymentId,
            'payment_transection_id' => $paymentDetails['payment_transection_id'],
            'payment_refferance_id' => $paymentDetails['payment_refferance_id'],
            'payment_transection_data' => $paymentDetails['payment_transection_data'],
            'payment_status' => $paymentDetails['payment_status'],
            'order_status_id' => $order_status_id,
            'updated_at' => now()
        ]);
    }
    private function deductFromWallet($user_id, $orderId, $amount)
    {
        UserWallet::insertGetId([
            'user_id' => $user_id,
            'order_id' => $orderId,
            'payment_id' => null,
            'type' => 'debit',
            'amount' => -$amount,
            'description' => 'We have debited ' . \Helpers::currency($amount) . ' from your wallet for a new purchase.',
            'created_at' => now()
        ]);
    }
    
    
    private function manage_payment_status_field($input)
    {
        if(isset($input['payment_status'])){
            $payment_status = $input['payment_status'];
        }else{
            $payment_status = 1;
        }
        
        if(isset($input['payment']['method'])){
             $payment_method = $input['payment']['method'];
        }else{
             $payment_method = '';
        }
        
       

        // Define status based on payment method and payment status
        if ($payment_status == 1) {

            if ($payment_method == 'cash') {
                $status = 'Unpaid';
            }else{
                $status = 'Paid';
            }

        } elseif ($payment_status == 2 || $payment_status == 3) {
            if ($payment_method != 'cash') {
                $status = 'Failed'; // Online payment with status 2 or 3
            }
        } else {
            $status = 'Processing'; // Fallback for any other cases
        }

        return $status;
    }

    private function sendNotifications($user_id, $orderId)
    {
        $key = setting('fcm_key');
        $tokens = [];
        $player_ids = [];

        // Retrieve user login data
        $users = UserAuthLogin::where('user_id', $user_id)->get();

        if ($users->count() > 0) {
            foreach ($users as $row) {
                if (setting('is_enable_one_signal')) {
                    if (!is_null($row->player_id)) {
                        $player_ids[] = $row->player_id;
                    }
                } else {
                    if (!is_null($row->fcm_token)) {
                        $tokens[] = $row->fcm_token;
                    }
                }
            }
        }

        // Send notification via OneSignal or Firebase
        if (setting('is_enable_one_signal')) {
            \Helpers::sendNotification($player_ids, "Order Placed successfully.", "Order Placed", $key, false, $orderId);
        } else {
            \Helpers::sendNotification($tokens, "Order Placed successfully.", "Order Placed", $key, false, $orderId);
        }
        
        
        /* firebase notification for admin */
        if( setting('is_enable_firebase') && setting('is_enable_firebase_notification') && setting('firebase_json_file')) {
            if(Storage::path('/public/firebase/firebase_credentials.json')) {
                $admin=User::where('type','admin')->first();
                $tokens=json_decode($admin->fcm_token,true);
                $admin->notify(new FcmNotification(
                    'New Order Received.# '.$orderId,
                    'Click to go to order details.',
                    [
                        'url' => url('/admin/orders/' . $orderId), // This will be used for redirection
                        'image' => url('/app-assets/images/logo.png')
                    ],
                    $tokens // Pass tokens here for multiple user
                ));
            }
        }
        /* end notification */

        // Insert notification into the database
        $notificationPost = [
            'user_id' => $user_id,
            'order_id' => $orderId,
            'lang_code' => 'en',
            'title' => "Order Placed",
            'message' => "Order Placed successfully.",
            'created_at' => now(),
            'read_status' => 0
        ];
        Notification::insert($notificationPost);
    }

    private function sendOrderEmails($user_id, $order)
    {
        try {
            $isEnableOrderEmailSetting = AppSetting::get_setting('is_enable_mail_order_receive');
            if ($isEnableOrderEmailSetting == 1) {
                $customer = User::find($user_id);
                $emailData = [
                    'order' => $order,
                    'customer' => $customer,
                ];
                
                // Get all admin emails
                $admins = User::where('role_id', 1)->pluck('email')->toArray(); // Use pluck for simplicity
                $adminEmails = array_filter($admins); // Remove any null or empty emails
                
                $customerEmail = $customer->email;
                $subject = 'New Order Received - Order ID: ' . $order->id;
                
                $status = \Mail::send('email.order', $emailData, function ($message) use ($customerEmail, $adminEmails, $subject) {
                    // Send to the customer
                    $message->to($customerEmail);
            
                    // Add admins as recipients (optional: use `cc` or `bcc`)
                    if (!empty($adminEmails)) {
                        $message->cc($adminEmails);
                    }
            
                    // Set the subject
                    $message->subject($subject);
                });
            }
        } catch (\Throwable $e) {
            \Log::error($e->getMessage());
        }
    }
    /* end of functions for used in API */

}